% Vergeer_etal_2021_replication
%
% Replicates part of:
%   Vergeer P., van Schaik Y., Sjerps M. (2021). Measuring calibration of likelihood-ratio systems: A comparison of four metrics, including a new metric devPAV. 
%   Forensic Science International, 321, article 110722. https://doi.org/10.1016/j.forsciint.2021.110722
%
% tested on Matlab R2020b

clear all;
close all;
clc;
addpath('./functions/');
rng(0);

%% Monte Carlo simulations
num_s = [50 100 300];
num_d = (num_s.^2 - num_s) / 2;
max_num_s = max(num_s);
max_num_d = max(num_d);
length_num_s = length(num_s);

sigma2 = [2 12 22 34];
sigma = sqrt(sigma2);
mu_s = sigma2/2;
mu_d = - mu_s;
length_sigma = length(sigma);

num_iter = 10000; % set to 10000

Cllr = NaN(num_iter, length_sigma, length_num_s);
Cllr_min = Cllr;
devPAV_metric = Cllr;
Cllr_Cllr_expected = Cllr;

parfor I_iter = 1:num_iter
        
    for I_sigma = 1:length_sigma
        % Gaussian different-source
        ln_LR_d = random('Normal', mu_d(I_sigma), sigma(I_sigma), [max_num_d 1]);
        % Gaussian same-source
        ln_LR_s = random('Normal', mu_s(I_sigma), sigma(I_sigma), [max_num_s 1]);
        
        for I_num = 1:length_num_s
            
            ln_LR_s_temp = ln_LR_s(1:num_s(I_num));
            ln_LR_d_temp = ln_LR_d(1:num_d(I_num));
            
            % Cllr
            Cllr(I_iter, I_sigma, I_num) = cllr(ln_LR_s_temp, ln_LR_d_temp);
            
            % Cllr_min
            ln_LR_PAV = pav_transform([], ln_LR_s_temp, ln_LR_d_temp);
            ln_LR_s_PAV = ln_LR_PAV(1:num_s(I_num));
            ln_LR_d_PAV = ln_LR_PAV(num_s(I_num)+1:end);
            Cllr_min(I_iter, I_sigma, I_num) = cllr(ln_LR_s_PAV, ln_LR_d_PAV);
            
            % devPAV
            range_min = min(ln_LR_s_temp);
            range_max = max(ln_LR_d_temp);
            ln_LR_temp = [ln_LR_s_temp; ln_LR_d_temp];
            II_range = (ln_LR_temp >= range_min) & (ln_LR_temp <= range_max);
            devPAV_metric(I_iter, I_sigma, I_num) = devPAV(ln_LR_temp(II_range), ln_LR_PAV(II_range));
            
        end
    end
end

% Cllr_cal
Cllr_cal = Cllr - Cllr_min;

% Cllr - Cllr_expected
num_expected = 500000;

Cllr_expected = NaN(length_sigma,1);
parfor I_sigma = 1:length_sigma
    % Gaussian different-source
    ln_LR_d_expected = random('Normal', mu_d(I_sigma), sigma(I_sigma), [num_expected 1]);
    % Gaussian same-source
    ln_LR_s_expected = random('Normal', mu_s(I_sigma), sigma(I_sigma), [num_expected 1]);
    
    Cllr_expected(I_sigma) = cllr(ln_LR_s_expected, ln_LR_d_expected);
    Cllr_Cllr_expected(:, I_sigma, :) = Cllr(:, I_sigma, :) - Cllr_expected(I_sigma);
end

fprintf('\nsigma2:\t%0.0f\tCllr_expected:\t%0.3f\n', [sigma2; Cllr_expected'])

%% plot figures

% Cllr - Cllr_expected
h_fig = figure(8);
y_lim = [-0.2 0.2];
y_label = '\itC\rm_{llr} – \itC\rm_{llr}^{expect}';
Vergeer_etal_2021_replication_plot(h_fig, Cllr_Cllr_expected, length_num_s, sigma2, y_lim, y_label);

% Cllr_cal
h_fig = figure(9);
y_lim = [0 0.12];
y_label = '\itC\rm_{llr}^{cal}';
Vergeer_etal_2021_replication_plot(h_fig, Cllr_cal, length_num_s, sigma2, y_lim, y_label);

% devPAV
h_fig = figure(10);
y_lim = [0 1.75];
y_label = 'devPAV';
Vergeer_etal_2021_replication_plot(h_fig, devPAV_metric, length_num_s, sigma2, y_lim, y_label);

% cleanup
rmpath('./functions/');